<?php
require_once __DIR__ . "/../models/Organizacion.php";


class OrganizacionController{
    private $organizacionModel;
    private $userModel;
    private $dataErrors = [];
    private $header;

    public function __construct($mysqli,$header){
        $this->organizacionModel = new Organizacion($mysqli,$header);
        $this->header = $header;
    }

    public function crearPuestoEmpresa($data){
        try {
            if($this->header['profile'] !='ADMIN') throw new Exception('E401');
            $idRegistro = $this->organizacionModel->crearPuestoEmpresa($data);
            if($idRegistro) ApiResponse::success('S002',$idRegistro); throw new Exception('E500');
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function crearDepartamentoEmpresa($data){
        try {
            if($this->header['profile'] !='ADMIN') throw new Exception('E401');
            $idRegistro = $this->organizacionModel->crearDepartamentoEmpresa($data);
            if($idRegistro) ApiResponse::success('S002',$idRegistro); throw new Exception('E500');
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function crearColaboradorEmpresa($data){
        try {
            if($this->header['profile'] !='ADMIN') throw new Exception('E401');
            if(!$this->validarRut($data['rut']."-".$data['dv'])) ApiResponse::error('E400',"RUT colaborador invalido");
            
            $idRegistro = $this->organizacionModel->crearColaboradorEmpresa($data);
            if($idRegistro) ApiResponse::success('S002',$idRegistro); throw new Exception('E500');
            
            
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function actualizarColaboradorEmpresa($data,$id){
        try {
            if ($this->header['profile'] != 'ADMIN') throw new Exception('E401');
            if (!$this->validarRut($data['rut'] . "-" . $data['dv'])) ApiResponse::error('E400', "RUT colaborador invalido");
            $filasAfectadas = $this->organizacionModel->actualizarColaboradorEmpresa($data, $id);
            if ($filasAfectadas > 0) {
                ApiResponse::success('S003', $id);
            } else {
                ApiResponse::success('S005');
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function actualizarPuestoTrabajoEmpresa($data, $id){
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            // Validar que al menos uno venga
            $nombre = $data['nombre'] ?? null;
            $descripcion = $data['descripcion'] ?? null;
            if ((empty($nombre) || $nombre === "") && (empty($descripcion) || $descripcion === "")) {
                ApiResponse::error('E400', 'Debe enviar al menos uno de los campos: nombre o descripcion');
            }
            $resp = $this->organizacionModel->actualizarPuestoTrabajoEmpresa($data, $id);
            if ($resp > 0) {
                ApiResponse::success('S003', $id);
            } else {
                ApiResponse::success('S005');
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function actualizarDepartamentoEmpresa($data, $id){
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            $nombre = $data['nombre'] ?? null;
            $idPadre = $data['id_departamento_padre'] ?? null;
            if ((empty($nombre) || $nombre === '') && ($idPadre === null || $idPadre === '')) {
                ApiResponse::error('E400', 'Debe enviar al menos uno de los campos: nombre o id_departamento_padre');
            }
            $resp = $this->organizacionModel->actualizarDepartamentoEmpresa($data, $id);
            if ($resp > 0) {
                ApiResponse::success('S003', $id);
            } else {
                ApiResponse::success('S005');
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function bloqueaColaboradorEmpresa($data,$id){
        try {
            if ($this->header['profile'] != 'ADMIN') throw new Exception('E401');
            if ($data['bloqueo'] === 'true'){
                $filasAfectadas = $this->organizacionModel->bloqueaColaboradorEmpresa($id);
            }else{
                $filasAfectadas = $this->organizacionModel->desbloqueaColaboradorEmpresa($id);
            }
            if ($filasAfectadas > 0) {
                ApiResponse::success('S003');
            } else {
                ApiResponse::success('S005');
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function eliminaColaboradorEmpresa($id){
        try {
            if ($this->header['profile'] != 'ADMIN') throw new Exception('E401');
            $filasAfectadas = $this->organizacionModel->eliminaColaboradorEmpresa($id);
            if ($filasAfectadas > 0) {
                ApiResponse::success('S004');
            } else {
                ApiResponse::success('S005');
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function eliminarPuestoTrabajoEmpresa($idPuesto){
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            $resp = $this->organizacionModel->eliminarPuestoTrabajoEmpresa($idPuesto);
            if ($resp > 0) {
                ApiResponse::success('S004', $resp); // S003: recurso eliminado
            } else {
                throw new Exception('E504'); // No se eliminó ningún registro
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function eliminarDepartamentoEmpresa($idDepartamento){
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            $resp = $this->organizacionModel->eliminarDepartamentoEmpresa($idDepartamento);
            if ($resp > 0) {
                ApiResponse::success('S004', $resp); // S003: Recurso eliminado
            } else {
                throw new Exception('E504'); // Nada se eliminó
            }
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function listarColaboradoresEmpresa($data) {
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            $result = $this->organizacionModel->listarColaboradorEmpresa($data);
            ApiResponse::success('S001', $result);
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function listarDepartamentosEmpresa() {
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            $result = $this->organizacionModel->listarDepartamentosEmpresa();
            ApiResponse::success('S001', $result);

        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function estructuraDepartamentosEmpresa() {
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }
            $result = $this->organizacionModel->estructuraDepartamentosEmpresa();
            ApiResponse::success('S001', $result);
        
        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }
    public function listarPuestosEmpresa(){
        try {
            if ($this->header['profile'] !== 'ADMIN') {
                throw new Exception('E401');
            }

            $result = $this->organizacionModel->listarPuestosEmpresa();
            ApiResponse::success('S001', $result);

        } catch (Exception $e) {
            ApiResponse::error($e->getMessage());
        }
    }



    private function formateaResumenGeneral($data){
        // Orden deseado de estados
        $order = ["TOTAL", "ACTIVO", "INACTIVO", "BLOQUEADO", "ELIMINADO"];

        // 1) Agrupar por tipo
        $grouped = [];
        foreach ($data as $row) {
            $tipo   = $row['tipo'];
            $estado = $row['estado'];
            $total  = (int)$row['total'];
        
            if (!isset($grouped[$tipo])) {
                $grouped[$tipo] = [];
            }
            // guardamos por estado para poder ordenar después
            $grouped[$tipo][$estado] = ['estado' => $estado, 'total' => $total];
        }

        // 2) Construir salida con estructura requerida y orden de estados
        $output = [];
        foreach ($grouped as $tipo => $estadosMap) {
            $detalle = [];
        
            // respetar el orden predefinido; incluir solo los estados presentes
            foreach ($order as $estadoDeseado) {
                if (isset($estadosMap[$estadoDeseado])) {
                    $detalle[] = $estadosMap[$estadoDeseado];
                }
            }
        
            // por si aparece algún estado no previsto en $order
            foreach ($estadosMap as $estado => $item) {
                if (!in_array($estado, $order, true)) {
                    $detalle[] = $item;
                }
            }
        
            $output[] = [
                'area'    => $tipo,
                'detalle' => $detalle
            ];
        }
        return $output;
    }

    private function validarRut($rutCompleto) {
        // Verificar formato: dígitos + guión + dígito verificador (0-9 o k/K)
        if (!preg_match('/^[0-9]+[-|‐]{1}[0-9kK]{1}$/', $rutCompleto)) {
            return false;
        }

        // Separar número y dígito verificador
        list($rut, $digv) = explode('-', $rutCompleto);

        // Calcular dígito verificador
        $M = 0;
        $S = 1;
        for ($T = $rut; $T > 0; $T = floor($T / 10)) {
            $S = ($S + ($T % 10) * (9 - $M++ % 6)) % 11;
        }
        $dvCalculado = $S ? $S - 1 : 'k';

        // Comparar dígito verificador (ignorar mayúsculas/minúsculas)
        return $dvCalculado == strtolower($digv);
    }
}